/*
 * IRC - Internet Relay Chat, ircd/m_fjoin.c
 * Copyright (C) 1990 Jarkko Oikarinen and
 *                    University of Oulu, Computing Center
 *
 * See file AUTHORS in IRC package for additional names of
 * the programmers.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 1, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: m_tmpl.c,v 1.4 2004/12/11 05:14:03 klmitch Exp $
 */

/*
 * m_functions execute protocol messages on this server:
 *
 *    cptr    is always NON-NULL, pointing to a *LOCAL* client
 *            structure (with an open socket connected!). This
 *            identifies the physical socket where the message
 *            originated (or which caused the m_function to be
 *            executed--some m_functions may call others...).
 *
 *    sptr    is the source of the message, defined by the
 *            prefix part of the message if present. If not
 *            or prefix not found, then sptr==cptr.
 *
 *            (!IsServer(cptr)) => (cptr == sptr), because
 *            prefixes are taken *only* from servers...
 *
 *            (IsServer(cptr))
 *                    (sptr == cptr) => the message didn't
 *                    have the prefix.
 *
 *                    (sptr != cptr && IsServer(sptr) means
 *                    the prefix specified servername. (?)
 *
 *                    (sptr != cptr && !IsServer(sptr) means
 *                    that message originated from a remote
 *                    user (not local).
 *
 *            combining
 *
 *            (!IsServer(sptr)) means that, sptr can safely
 *            taken as defining the target structure of the
 *            message in this server.
 *
 *    *Always* true (if 'parse' and others are working correct):
 *
 *    1)      sptr->from == cptr  (note: cptr->from == cptr)
 *
 *    2)      MyConnect(sptr) <=> sptr == cptr (e.g. sptr
 *            *cannot* be a local connection, unless it's
 *            actually cptr!). [MyConnect(x) should probably
 *            be defined as (x == x->from) --msa ]
 *
 *    parc    number of variable parameter strings (if zero,
 *            parv is allowed to be NULL)
 *
 *    parv    a NULL terminated list of parameter pointers,
 *
 *                    parv[0], sender (prefix string), if not present
 *                            this points to an empty string.
 *                    parv[1]...parv[parc-1]
 *                            pointers to additional parameters
 *                    parv[parc] == NULL, *always*
 *
 *            note:   it is guaranteed that parv[0]..parv[parc-1] are all
 *                    non-NULL pointers.
 */
#include "config.h"

#include "channel.h"
#include "client.h"
#include "hash.h"
#include "ircd.h"
#include "ircd_log.h"
#include "ircd_reply.h"
#include "ircd_string.h"
#include "msg.h"
#include "numeric.h"
#include "numnicks.h"
#include "send.h"
#include "s_user.h"

/* #include <assert.h> -- Now using assert in ircd_log.h */

/*
 * ms_fjoin - server message handler
 */
int ms_fjoin(struct Client* cptr, struct Client* sptr, int parc, char* parv[])
{
  struct Channel *chptr;
  struct Client *acptr;
  char *name;
  unsigned int flags = CHFL_DEOPPED;
  time_t creation = 0;
  char *p = 0;

  if (parc < 3)
    return need_more_params(sptr, "FJOIN");

  if (!IsPrivileged(sptr)) {
    protocol_violation(sptr, "No privileges on source for FJOIN, desync?");
    return send_reply(sptr, ERR_NOPRIVILEGES);
  }

  if (!(acptr = findNUser(parv[2])) || !IsUser(acptr))
    return 0;

  if (parc > 3 && *parv[3]) {
    creation = atoi(parv[3]);

    /* it's actually an error return, tell them */
    if (creation == 0) {
      if (MyUser(acptr))
	send_reply(acptr, ERR_INVITEONLYCHAN, parv[1]);

      /* tell the others about it */
      sendcmdto_serv_butone(sptr, CMD_FJOIN, cptr, "%s %s 0", parv[1], parv[2]);
      return 0;
    }
  }

  for (name = ircd_strtok(&p, parv[1], ","); name;
       name = ircd_strtok(&p, 0, ",")) {

    if (IsLocalChannel(name) || !IsChannelName(name))
    {
      protocol_violation(cptr, "%s tried to join %s", cli_name(sptr), name);
      continue;
    }

    if (chptr = FindChannel(name)) {
      /* join to existing channel */
      sendcmdto_channel_butserv_butone(acptr, CMD_JOIN, chptr, NULL, 0, "%H",
				       chptr);
    } else {
      /* create new channel */
      chptr = get_channel(acptr, name, CGT_CREATE);
      chptr->creationtime = creation;
    }

    int oplevel = !chptr->mode.apass[0] ? MAXOPLEVEL
	: (flags & CHFL_CHANNEL_MANAGER) ? 0
	: 1;
    if ((chptr->mode.mode & MODE_DELJOINS) && !(flags & CHFL_VOICED_OR_OPPED))
      add_user_to_channel(chptr, acptr, flags | CHFL_DELAYED, oplevel);
    else
      add_user_to_channel(chptr, acptr, flags, oplevel);

    if (MyUser(acptr)) {
      sendcmdto_one(acptr, CMD_JOIN, acptr, ":%H", chptr);
      del_invite(acptr, chptr);

      if (chptr->topic[0]) {
	send_reply(acptr, RPL_TOPIC, chptr->chname, chptr->topic);
	send_reply(acptr, RPL_TOPICWHOTIME, chptr->chname, chptr->topic_nick,
		   chptr->topic_time);
      }

      do_names(acptr, chptr, NAMES_ALL|NAMES_EON); /* send /names list */
    }
  }

  /* broadcast */
  sendcmdto_serv_butone(sptr, CMD_FJOIN, cptr, "%s %s %Tu", parv[1], parv[2], creation);
}

